//
//  KeyCommands.swift
//  Do It
//
//  Created by Jim Dovey on 4/1/20.
//  Copyright © 2020 Jim Dovey. All rights reserved.
//

import UIKit
import SwiftUI
import Combine

struct KeyCommand: Hashable {
    private let _command: UIKeyCommand
    
    fileprivate init?(_ uikit: UIKeyCommand?) {
        guard let uikit = uikit else { return nil }
        self._command = uikit
    }
    
    init(title: String, input: String, modifierFlags: ModifierFlags = [],
         attributes: Attributes = [], discoverabilityTitle: String? = nil) {
        self._command = UIKeyCommand(
            title: title,
            action: #selector(UIResponder.swiftUIKeyCommand(_:)),
            input: input,
            modifierFlags: UIKeyModifierFlags(rawValue: modifierFlags.rawValue),
            discoverabilityTitle: discoverabilityTitle,
            attributes: UIMenuElement.Attributes(rawValue: attributes.rawValue)
        )
    }
    
    fileprivate var uikit: UIKeyCommand { _command }
    
    var input: String? { _command.input }
    var modifierFlags: UIKeyModifierFlags { _command.modifierFlags }
    var title: String { _command.title }
    var discoverabilityTitle: String? { _command.discoverabilityTitle }
    
    struct ModifierFlags: OptionSet {
        var rawValue: Int
        
        init(rawValue: Int) {
            self.rawValue = rawValue
        }
        private init(_ uikit: UIKeyModifierFlags) {
            self.rawValue = uikit.rawValue
        }
        
        static var alphaShift = ModifierFlags(.alphaShift)
        static var shift = ModifierFlags(.shift)
        static var control = ModifierFlags(.control)
        static var alternate = ModifierFlags(.alternate)
        static var command = ModifierFlags(.command)
        static var numericPad = ModifierFlags(.numericPad)
    }
    
    struct Attributes: OptionSet {
        var rawValue: UInt
        
        init(rawValue: UInt) {
            self.rawValue = rawValue
        }
        private init(_ uikit: UIMenuElement.Attributes) {
            self.rawValue = uikit.rawValue
        }
        
        static var disabled = Attributes(.disabled)
        static var destructive = Attributes(.destructive)
        static var hidden = Attributes(.hidden)
    }
    
    static func == (lhs: KeyCommand, rhs: KeyCommand) -> Bool {
        return lhs._command == rhs._command
    }
    
    func hash(into hasher: inout Hasher) {
        hasher.combine(_command)
    }
}


// Defines the action targeted by SwiftUI key commands, so that
// #selector() works.
fileprivate extension UIResponder {
    @objc func swiftUIKeyCommand(_ sender: UIKeyCommand?) {}
}

// MARK: -
